/*
 * Decompiled with CFR 0.152.
 */
package org.tinymediamanager.core;

import java.awt.Point;
import java.awt.image.BufferedImage;
import java.awt.image.BufferedImageOp;
import java.awt.image.ColorConvertOp;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.security.MessageDigest;
import java.util.ArrayList;
import java.util.List;
import javax.imageio.IIOImage;
import javax.imageio.ImageIO;
import javax.imageio.ImageWriteParam;
import javax.imageio.ImageWriter;
import javax.imageio.stream.FileImageOutputStream;
import org.imgscalr.Scalr;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tinymediamanager.Globals;
import org.tinymediamanager.core.EmptyFileException;
import org.tinymediamanager.core.ImageUtils;
import org.tinymediamanager.core.InvalidFileTypeException;
import org.tinymediamanager.core.Settings;
import org.tinymediamanager.core.Utils;
import org.tinymediamanager.core.entities.MediaEntity;
import org.tinymediamanager.core.entities.MediaFile;
import org.tinymediamanager.scraper.http.Url;
import org.tinymediamanager.scraper.util.StrgUtils;
import org.tinymediamanager.scraper.util.UrlUtil;

public class ImageCache {
    private static final Logger LOGGER = LoggerFactory.getLogger(ImageCache.class);
    private static final Path CACHE_DIR = Paths.get(Globals.CACHE_FOLDER + "/image", new String[0]);
    private static final char[] HEX_DIGITS = "0123456789ABCDEF".toCharArray();

    public static void createSubdirs() {
        if (!Files.exists(CACHE_DIR, new LinkOption[0])) {
            try {
                Files.createDirectories(CACHE_DIR, new FileAttribute[0]);
            }
            catch (IOException e) {
                LOGGER.warn("Could not create cache dir {} - {}", (Object)CACHE_DIR, (Object)e.getMessage());
            }
        }
        for (char sub : HEX_DIGITS) {
            try {
                Path p = CACHE_DIR.resolve(Character.toString(sub));
                Files.createDirectories(p, new FileAttribute[0]);
            }
            catch (FileAlreadyExistsException p) {
            }
            catch (IOException e) {
                LOGGER.warn("Could not create cache sub dir '{}' - {}", (Object)Character.valueOf(sub), (Object)e.getMessage());
            }
        }
    }

    @Deprecated
    public static void migrate() {
        List<Path> files = Utils.listFiles(ImageCache.getCacheDir());
        for (Path f : files) {
            Path f2 = ImageCache.getCacheDir().resolve(Paths.get(f.getFileName().toString().substring(0, 1), f.getFileName().toString()));
            try {
                Files.move(f, f2, StandardCopyOption.ATOMIC_MOVE);
            }
            catch (IOException iOException) {}
        }
    }

    public static Path getCacheDir() {
        return CACHE_DIR;
    }

    public static String getMD5(String path) {
        try {
            if (path == null) {
                return null;
            }
            MessageDigest md = MessageDigest.getInstance("MD5");
            byte[] key = md.digest(path.getBytes());
            return StrgUtils.bytesToHex((byte[])key);
        }
        catch (Exception e) {
            LOGGER.error("Failed to create cached filename for image: {} - {}", (Object)path, (Object)e);
            return "";
        }
    }

    public static String getMD5WithSubfolder(String path) {
        String md5 = ImageCache.getMD5(path);
        if (path == null) {
            return null;
        }
        return Paths.get(md5.substring(0, 1), md5).toString();
    }

    public static Path cacheImage(MediaFile mediaFile) throws Exception {
        return ImageCache.cacheImage(mediaFile, false);
    }

    private static Path cacheImage(MediaFile mediaFile, boolean overwrite) throws Exception {
        if (!mediaFile.isGraphic()) {
            throw new InvalidFileTypeException(mediaFile.getFileAsPath());
        }
        Path originalFile = mediaFile.getFileAsPath();
        Path cachedFile = ImageCache.getCacheDir().resolve(ImageCache.getMD5WithSubfolder(originalFile.toString()) + "." + Utils.getExtension(originalFile));
        if (overwrite || !Files.exists(cachedFile, new LinkOption[0])) {
            if (!Files.exists(originalFile, new LinkOption[0])) {
                throw new FileNotFoundException("unable to cache file: " + originalFile + "; file does not exist");
            }
            if (Files.size(originalFile) == 0L) {
                throw new EmptyFileException(originalFile);
            }
            BufferedImage originalImage = null;
            int retries = 5;
            while (true) {
                try {
                    originalImage = ImageUtils.createImage(originalFile);
                }
                catch (OutOfMemoryError e) {
                    LOGGER.warn("hit memory cap: {}", (Object)e.getMessage());
                    Thread.sleep(500L);
                    if (--retries > 0) continue;
                }
                break;
            }
            int desiredWidth = originalImage.getWidth();
            if ((float)originalImage.getWidth() / (float)originalImage.getHeight() > 1.0f) {
                if (originalImage.getWidth() > 400) {
                    desiredWidth = 400;
                }
            } else if (originalImage.getHeight() > 400) {
                desiredWidth = 400 * originalImage.getWidth() / originalImage.getHeight();
            }
            Point size = ImageUtils.calculateSize(desiredWidth, (int)((double)originalImage.getHeight() / 1.5), originalImage.getWidth(), originalImage.getHeight(), true);
            BufferedImage scaledImage = null;
            retries = 5;
            while (true) {
                try {
                    if (Globals.settings.getImageCacheType() == CacheType.FAST) {
                        scaledImage = Scalr.resize((BufferedImage)originalImage, (Scalr.Method)Scalr.Method.BALANCED, (Scalr.Mode)Scalr.Mode.FIT_EXACT, (int)size.x, (int)size.y, (BufferedImageOp[])new BufferedImageOp[0]);
                        break;
                    }
                    scaledImage = Scalr.resize((BufferedImage)originalImage, (Scalr.Method)Scalr.Method.QUALITY, (Scalr.Mode)Scalr.Mode.FIT_EXACT, (int)size.x, (int)size.y, (BufferedImageOp[])new BufferedImageOp[0]);
                }
                catch (OutOfMemoryError e) {
                    LOGGER.warn("hit memory cap: {}", (Object)e.getMessage());
                    Thread.sleep(500L);
                    if (--retries > 0) continue;
                }
                break;
            }
            originalImage = null;
            ImageWriter imgWrtr = null;
            ImageWriteParam imgWrtrPrm = null;
            if (ImageUtils.hasTransparentPixels(scaledImage)) {
                imgWrtr = ImageIO.getImageWritersByFormatName("png").next();
                imgWrtrPrm = imgWrtr.getDefaultWriteParam();
            } else {
                BufferedImage rgb = new BufferedImage(scaledImage.getWidth(), scaledImage.getHeight(), 1);
                ColorConvertOp xformOp = new ColorConvertOp(null);
                xformOp.filter(scaledImage, rgb);
                imgWrtr = ImageIO.getImageWritersByFormatName("jpg").next();
                imgWrtrPrm = imgWrtr.getDefaultWriteParam();
                imgWrtrPrm.setCompressionMode(2);
                imgWrtrPrm.setCompressionQuality(0.8f);
                scaledImage = rgb;
            }
            FileImageOutputStream output = new FileImageOutputStream(cachedFile.toFile());
            imgWrtr.setOutput(output);
            IIOImage image = new IIOImage(scaledImage, null, null);
            imgWrtr.write(null, image, imgWrtrPrm);
            imgWrtr.dispose();
            output.flush();
            output.close();
            scaledImage = null;
            if (!Files.exists(cachedFile, new LinkOption[0])) {
                throw new Exception("unable to cache file: " + originalFile);
            }
        }
        return cachedFile;
    }

    public static void cacheImageSilently(Path path) {
        ImageCache.cacheImageSilently(new MediaFile(path));
    }

    public static void cacheImageSilently(MediaFile mediaFile) {
        if (!Settings.getInstance().isImageCache()) {
            return;
        }
        if (!mediaFile.isGraphic()) {
            return;
        }
        try {
            ImageCache.cacheImage(mediaFile, true);
        }
        catch (Exception e) {
            LOGGER.warn("could not cache image: {}", (Object)e.getMessage());
        }
    }

    public static void invalidateCachedImage(Path path) {
        ImageCache.invalidateCachedImage(new MediaFile(path));
    }

    public static void invalidateCachedImage(MediaFile mediaFile) {
        if (!mediaFile.isGraphic()) {
            return;
        }
        Path path = mediaFile.getFileAsPath();
        Path cachedFile = ImageCache.getCacheDir().resolve(ImageCache.getMD5WithSubfolder(path.toAbsolutePath().toString()) + "." + Utils.getExtension(path));
        if (Files.exists(cachedFile, new LinkOption[0])) {
            Utils.deleteFileSafely(cachedFile);
        }
    }

    public static Path getCachedFile(String url) {
        Path cachedFile;
        if (url == null || url.isEmpty()) {
            return null;
        }
        String ext = UrlUtil.getExtension((String)url);
        if (ext.isEmpty()) {
            ext = "jpg";
        }
        if (Files.exists(cachedFile = ImageCache.getCacheDir().resolve(ImageCache.getMD5WithSubfolder(url) + "." + ext), new LinkOption[0])) {
            LOGGER.trace("found cached url :) {}", (Object)url);
            return cachedFile;
        }
        if (!Globals.settings.isImageCache()) {
            return null;
        }
        try {
            LOGGER.trace("downloading image to the image cache: {}", (Object)url);
            Url u = new Url(url);
            boolean ok = u.download(cachedFile);
            if (ok) {
                LOGGER.trace("cached url successfully :) {}", (Object)url);
                return cachedFile;
            }
        }
        catch (Exception e) {
            LOGGER.trace("Problem getting cached file for url {}", (Object)e.getMessage());
        }
        LOGGER.trace("could not get cached file for url {}", (Object)url);
        return null;
    }

    public static Path getCachedFile(Path path) {
        return ImageCache.getCachedFile(new MediaFile(path));
    }

    public static Path getCachedFile(MediaFile mediaFile) {
        if (mediaFile == null || !mediaFile.isGraphic()) {
            return null;
        }
        Path path = mediaFile.getFileAsPath().toAbsolutePath();
        Path cachedFile = ImageCache.getCacheDir().resolve(ImageCache.getMD5WithSubfolder(path.toString()) + "." + Utils.getExtension(path));
        if (Files.exists(cachedFile, new LinkOption[0])) {
            LOGGER.trace("found cached file :) {}", (Object)path);
            return cachedFile;
        }
        if (path.startsWith(CACHE_DIR.toAbsolutePath())) {
            return path;
        }
        if (!Globals.settings.isImageCache()) {
            LOGGER.trace("ImageCache not activated!");
            return null;
        }
        try {
            Path p = ImageCache.cacheImage(mediaFile);
            LOGGER.trace("cached file successfully :) {}", (Object)p);
            return p;
        }
        catch (EmptyFileException e) {
            LOGGER.warn("failed to cache file (file is empty): {}", (Object)path);
        }
        catch (FileNotFoundException e) {
        }
        catch (Exception e) {
            LOGGER.warn("problem caching file: {}", (Object)e.getMessage());
        }
        return null;
    }

    public static boolean isImageCached(Path path) {
        if (!Globals.settings.isImageCache()) {
            return false;
        }
        Path cachedFile = CACHE_DIR.resolve(ImageCache.getMD5WithSubfolder(path.toString()) + "." + Utils.getExtension(path));
        return Files.exists(cachedFile, new LinkOption[0]);
    }

    public static void clearImageCacheForMediaEntity(MediaEntity entity) {
        ArrayList<MediaFile> mediaFiles = new ArrayList<MediaFile>(entity.getMediaFiles());
        for (MediaFile mediaFile : mediaFiles) {
            Path file;
            if (!mediaFile.isGraphic() || (file = ImageCache.getCachedFile(mediaFile)) == null) continue;
            Utils.deleteFileSafely(file);
        }
    }

    static {
        ImageCache.createSubdirs();
    }

    public static enum CacheType {
        FAST,
        SMOOTH;

    }
}

